"""
Screenshot OCR Tool - Extracts actual text from images
Uses Tesseract OCR (free and open-source)
"""

from PIL import Image
import pytesseract
import cv2
import numpy as np

def preprocess_image(image_path):
    """Preprocess image for better OCR results"""
    try:
        # Read image
        img = cv2.imread(image_path)
        
        # Convert to grayscale
        gray = cv2.cvtColor(img, cv2.COLOR_BGR2GRAY)
        
        # Apply thresholding to get better contrast
        gray = cv2.threshold(gray, 0, 255, cv2.THRESH_BINARY | cv2.THRESH_OTSU)[1]
        
        # Denoise
        gray = cv2.medianBlur(gray, 3)
        
        return gray
    except:
        # Fallback to PIL
        return np.array(Image.open(image_path).convert('L'))

def extract_text_from_screenshot(image_path):
    """
    Extract text from screenshot using Tesseract OCR
    Returns actual extracted text or "No text detected" message
    """
    try:
        # Preprocess image
        processed_img = preprocess_image(image_path)
        
        # Extract text using Tesseract
        # PSM 3 = Fully automatic page segmentation
        text = pytesseract.image_to_string(processed_img, config='--psm 3')
        
        # Clean up text
        text = text.strip()
        
        # Try alternative method if no text found
        if not text or len(text) < 3:
            # Try with original image
            text = pytesseract.image_to_string(
                Image.open(image_path), 
                config='--psm 6'  # Assume uniform block of text
            ).strip()
        
        # Return result
        if not text or len(text) < 2:
            return "No text detected in the uploaded screenshot."
        
        return text
        
    except Exception as e:
        return f"OCR processing error: {str(e)}"

# For testing
if __name__ == "__main__":
    pass
